<?php
/**
 * Created by IntelliJ IDEA.
 * User: Loïc Rameau
 * Date: 11/05/2016
 * Time: 15:48
 */

namespace Bloom\Framework;

use Assetic\Extension\Twig\AsseticExtension;
use Bloom\Configuration;
use Bloom\Controller\Request\Request;
use Bloom\Database\Database;
use Bloom\Framework\View\BView;
use Bloom\Singleton;

class Application
{
    use Singleton;

    /**
     * @var \Twig_Environment
     */
    protected $twig;
    protected $namespace = 'Bloom\Framework\\';

    /**
     * @var Modules[]
     */
    public static $modules = [];

    protected $regexP = '~^(?<mandatory>[^\[]+)?(\[(?<optional>.+)?\])?~';
    protected $regexP_named = '~{(?<name>\w+)(:(?<regx>[^}]+))?}~';

    /**
     * @var Configuration
     */
    public $configuration;
    /**
     */
    public $log;
    /**
     * @var Database
     */
    public $db;

    /**
     * @var \Twig_LoaderInterface
     */
    protected $loader;

    protected function __construct()
    {
        $this->loader = new \Twig_Loader_Filesystem(APP_PATH.'/views');
        $this->twig = new \Twig_Environment($this->loader, [
            'cache'=> APP_ENV === 'prod' ? CACHE_PATH : false
        ]);
        $this->configuration = Configuration::getInstance()->SetFile(CONF_PATH.'/'.APP_ENV)->HaveSection();
        $this->db = new Database(
            'mysql:host='.Configuration::get('host','BDD').';dbname='.Configuration::get('dbname','BDD'),
            Configuration::get('user','BDD'),
            Configuration::get('passwd','BDD'),
            [\PDO::MYSQL_ATTR_INIT_COMMAND => 'SET NAMES utf8']
        );
        $this->initSession();
        $this->genFilter();
        $this->genTest();
        $this->genFunction();
    }


    protected function genFunction()
    {
        $this->twig->addFunction(new \Twig_SimpleFunction('pop_errors', function(){
            $_SESSION[APP_NAME][APP_ENV]['error'] = [];
        }));
        $this->twig->addFunction(new \Twig_SimpleFunction('form_token', function($salt = null) {
            if ($salt === null) {
                $salt = SID.'-'.date('Y-m-d H:i:s');
            }
            $hash = md5(mt_rand(1, 1000000) . $salt);
            Application::setSession('csrf_hash', $hash);
            return '<input type="hidden" name="'. $hash .'" value="1"/>';
        }, ['pre_escaped'=>'html', 'is_safe'=>['html']]));
    }

    protected function genFilter()
    {
        $this->twig->addFilter(new \Twig_SimpleFilter('makeUri', function($string) {
            if (!preg_match('/^\//', $string)) {
                $string = '/'.$string;
            }
            switch(APP_ENV) {
                case 'prod':
                    return $string;
                    break;
                case 'preprod':
                    return $string;
                    break;
                case 'development' :
                default:
                    if (strpos(APP_PATH,'/app')) {
                        return '/app'.$string;
                    } else {
                        return '/dist'.$string;
                    }
            }
        }));
        $this->twig->addFilter(new \Twig_SimpleFilter('ENV', function($string) {
            return str_replace(BASE_PATH,'',constant($string));
        }));
    }

    protected function genTest()
    {
        $this->twig->addTest(new \Twig_SimpleTest('url_active', function($url) {
            return preg_match('~^'.$url.'~', $_SERVER['REQUEST_URI']);
        }));
    }

    public function display($view)
    {
        $views = explode('/',$view);
        $view = $views[0];
        unset($views[0]);

        if ($view === 'logout') {
            session_destroy();
            static::navigate('/home');
        }

        $tpl = '';
        if (array_key_exists('tpl', $_POST)) {
            $tpl = $_POST['tpl'];
        }

        foreach(static::$modules as $module)
        {
            if($module->getUrl() === '/'.$view) {
                $module->setTpl($tpl?:implode('/',$views));
                return $module->execute();
            }
        }

        $controller_class = $this->namespace.'Controller\\'.ucfirst($view).'Controllers';
        if (class_exists($controller_class)) {
            $controller = new $controller_class($this);
            $controller->setTpl($tpl?:implode('/',$views));
            return $controller->execute();
        }
        $view_class = $this->namespace.'View\\'.ucfirst($view).'View';

        if (!class_exists($view_class)) {
            return static::renderS('error.twig',['error_msg'=>'404 Page Not Found']);
        }
        /**
         * @var BView $view
         */
        $view = new $view_class();
        if(count($views)>0) {
            preg_match($this->regexP, $view_class::$parameter, $matches);
            $parameter = $this->parse_parameter($matches, '/'.implode($views));
            $view->setParameter($parameter);
        }
        return $view->display($tpl);
    }

    /**
     * @param string $template
     * @param array  $data
     * @return string
     */
    public static function renderS($template, array $data = [])
    {
        return static::getInstance()->render($template, $data);
    }

    /**
     * @param string $template
     * @param array  $data
     * @return string
     */
    public function render($template, array $data = [])
    {
        return $this->twig->render($template, $data);
    }

    public function setNameSpace($namespace)
    {
        $this->namespace = $namespace;
    }

    public function parse_parameter(array $matches, $url)
    {
        $mandatory = $matches['mandatory'];
        $optional = $matches['optional'];
        $parameter = [];
        preg_match($this->regexP_named, $mandatory, $matches_tmp);
        if ($matches_tmp) {
            if(! preg_match('~'.str_replace($matches_tmp,'(?<value>'.$matches_tmp['regx'].')',$mandatory).'~', $url, $values) ){
                throw new \InvalidArgumentException('Invalid url exception');
            }
            $parameter[$matches_tmp['name']] = $values['value'];
        }
        preg_match($this->regexP_named, $optional, $matches_tmp);
        if ($matches_tmp) {
            if(! preg_match('~'.str_replace('['.$matches_tmp[0].']','(?<value>'.$matches_tmp['regx'].')',$optional).'~', $url, $values) ){
                throw new \InvalidArgumentException('Invalid url exception');
            }
            $parameter[$matches_tmp['name']] = $values['value'];
        }
        return $parameter;
    }

    public function initSession()
    {
        session_start();
        session_regenerate_id();
        if (!array_key_exists(APP_NAME, $_SESSION)) {
            $_SESSION[APP_NAME] = [
                APP_ENV => [
                    'session_start' => time(),
                    'error'=>[]
                ]
            ];
        }
    }

    /**
     * Get session variable
     * @param string $variable
     * @param null $default
     * @return mixed|null
     */
    public static function getSession($variable, $default = null)
    {
        if (! array_key_exists($variable, $_SESSION[APP_NAME][APP_ENV])) {
            return $default;
        }

        return $_SESSION[APP_NAME][APP_ENV][$variable];
    }

    /**
     * Set a session variable
     * @param string $variable
     * @param mixed $value
     */
    public static function setSession($variable, $value)
    {
        $_SESSION[APP_NAME][APP_ENV][$variable] = $value;
    }

    /**
     * @param string $url
     */
    public static function navigate($url)
    {
        header('Location: '.$url);
        exit(0);
    }

    /**
     * @param Modules $controller
     */
    public function registerModules( Modules $controller)
    {
        if($controller->getFactory() !== null) {
            $this->twig->addExtension(new AsseticExtension($controller->getFactory()));
        }
        $path = $controller->getViewPath();
        if ($path !== '') {
            if (is_array($path)) {
                foreach($path as $key=>$value) {
                    if (is_numeric($key)) {
                        $this->loader->addPath($value);
                    } else {
                        $this->loader->addPath($value, $key);
                    }
                }
            } else {
                $this->loader->addPath($path);
            }
        }
        static::$modules[] = $controller;
    }

    public static function reportError($message, $detail)
    {
        $_SESSION[APP_NAME][APP_ENV]['error'][] = ['message'=>$message, 'detail'=>$detail];
    }

    public static function validFormHash()
    {
        return Request::getInstance()->get(static::getSession('csrf_hash'), false) === '1';
    }

    public static function isCli()
    {
        return php_sapi_name() === 'cli';
    }

    /**
     * @return \Twig_Environment
     */
    public function getTwig()
    {
        return $this->twig;
    }

    /**
     * @return \Twig_Loader_Filesystem|\Twig_LoaderInterface
     */
    public function getLoader()
    {
        return $this->loader;
    }
}
