<?php

namespace Bloom\CMS\Core\Http;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

/**
 * Class Dossier
 *
 * @package Bloom\CMS\Core\Http
 *          --- Model ---
 * @property string               $name
 * @property string               $slug
 * @property string               $type
 * @property int                  parent_id
 * @property bool                 noindex
 *          --- Computed ---
 * @property string               $full_pathname
 * @property Page                 $index
 * @property int                  nb_parents
 *          -- Relation ---
 * @property Dossier              $parent
 * @property Dossier[]|Collection $childs
 * @property Page[]|Collection    $pages
 */
class Dossier extends Model
{
    protected $appends = ['full_pathname'];

    protected $guarded = [];

    protected static function booted()
    {
        static::updated(
            function (Dossier $dossier) {
                if ($dossier->isDirty(['slug', 'parent_id'])) {
                    $dossier->updatePathname();
                }
            }
        );
    }

    public function updatePathname()
    {
        foreach ($this->pages as $page) {
            $page->updatePathname();
            $page->save();
            if ($page->is_index && $page->dossier_id != $this->id) {
                $page->dossier->updatePathname();
            }
        }
    }

    public function parent()
    {
        return $this->belongsTo(Dossier::class, 'parent_id');
    }

    public function childs()
    {
        return $this->hasMany(Dossier::class, 'parent_id');
    }

    public function pages()
    {
        return $this->hasMany(Page::class);
    }

    public function miroirs()
    {
        return $this->hasMany(Miroir::class);
    }

    public function getIndexAttribute()
    {
        return $this->pages()->where('is_index', '=', 1)->first();
    }

    public function setSlugAttribute(string $slug)
    {
        $this->attributes['slug'] = Str::slug(str_replace('/', '', $slug));
    }

    public function getFullPathnameAttribute()
    {
        $slugs = [$this->slug];
        $parent = $this->parent;
        while ($parent !== null) {
            $slugs[] = $parent->slug;
            $parent = $parent->parent;
        }

        return implode('/', array_reverse($slugs));
    }

    public function getNbParentsAttribute()
    {
        if ($this->parent_id === null) {
            return 0;
        }

        return $this->parent->nb_parents + 1;
    }
}
