<?php

/**
 * Created by IntelliJ IDEA.
 * User: loic
 * Date: 02/12/2020
 * Time: 14:13
 */

namespace Tests\Unit;

use Bloom\CMS\Core\CmsCoreProvider;
use Bloom\CMS\Core\Contenus\Applicabilite;
use Bloom\CMS\Core\Http\Dossier;
use Bloom\CMS\Core\Http\Page;
use Bloom\Test\Article;
use Faker\Factory;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Foundation\Auth\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

/**
 * Class PageTest
 * @package Tests\Unit
 */
class PageTest extends \Bloom\CMS\Core\Test
{
    protected $provider = CmsCoreProvider::class;

    public static function setUpBeforeClass(): void
    {
        parent::setUpBeforeClass();
        require_once dirname(__DIR__, 2) . '/resources/module_test/src/Article.php';
    }

    protected function setUp(): void
    {
        parent::setUp();
        // On a besoin de s'authentifié car Page fixe lui même le created_by et last_modified_by
        Auth::setUser(User::first());
    }


    public function test_edit_route_should_be_empty_for_exceptions()
    {
        /**
         * @var Page $page
         */
        $page = factory(Page::class)->create();
        DB::table('pages')->where('id', '=', $page->id)->update(['contenu_type' => 'contact']);
        $page->refresh();

        static::assertEquals('#', $page->edit_route);
    }

    public function getArticle(Page $page): Article
    {
        $article = new Article();
        $article->contenu = "<p>test</p>";
        $article->save();
        $page->contenu()->associate($article);
        $page->save();

        return $article;
    }

    public function test_edit_route_should_check_for_applicability()
    {
        Page::declareContenu(Article::class);
        /**
         * @var Page $page
         */
        $page = factory(Page::class)->make();
        $a = $this->getArticle($page);

        $applicabilite = new Applicabilite();
        $applicabilite->titre = 'Article test applicable';
        $applicabilite->multiplicite = '1';
        $applicabilite->module_id = '1';
        $applicabilite->b_on = Applicabilite::ARCHI;
        $applicabilite->contenu_type = Article::class;
        $applicabilite->route = "admin_article_edit";
        $applicabilite->save();

        static::assertEquals(route('admin_article_edit', $a), $page->edit_route);
        $applicabilite->forceDelete();
    }

    public function test_edit_route_should_return_empty_if_note_applicable()
    {
        Page::declareContenu(Article::class);
        /**
         * @var Page $page
         */
        $page = factory(Page::class)->make();
        $this->getArticle($page);

        $applicabilite = new Applicabilite();
        $applicabilite->titre = 'Article test non applicable';
        $applicabilite->multiplicite = '1';
        $applicabilite->module_id = '1';
        $applicabilite->b_on = Applicabilite::STICKY;
        $applicabilite->contenu_type = Article::class;
        $applicabilite->route = "admin_article_edit";
        $applicabilite->save();

        static::assertEquals('#', $page->edit_route);
        $applicabilite->forceDelete();
    }

    public function test_edit_route_should_return_js_event_for_indexes()
    {
        $d = factory(Dossier::class)->create();
        /**
         * @var Page $page
         */
        $page = factory(Page::class)
            ->state('index')
            ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);

        static::assertStringContainsString('CustomEvent("edit-sommaire"', $page->edit_route);
    }

    public function test_has_publish_children_should_check_in_the_directory()
    {
        $d = factory(Dossier::class)->create();
        /**
         * @var Page $page
         */
        $page = factory(Page::class)
            ->state('index')
            ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);

        // On génére 1 page publié et 5 non publié
        /**
         * @var Page $p
         */
        $p = factory(Page::class)
            ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);
        factory(Page::class, 5)
            ->state('unpublished')
            ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);

        static::assertTrue($page->have_published_children);
        $p->unpublish();
        // On a besoin de reset la relation sinon il ne vois pas les changement
        $page->unsetRelations();
        static::assertFalse($page->have_published_children);
    }

    public function test_have_published_children_should_return_false_for_non_index()
    {
        $d = factory(Dossier::class)->create();
        /**
         * @var Page $page
         */
        $page = factory(Page::class)
            ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);
        factory(Page::class, 5)
            ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);

        static::assertFalse($page->have_published_children);
    }

    public function pageProvider()
    {
        $this->setUp();
        $faker = Factory::create();

        /**
         * @var Page[]|Collection $pages
         */
        $pages = factory(Page::class, 50)->state('index')->make();
        foreach ($pages as $page) {
            $d = factory(Dossier::class)->create();
            $page->dossier()->associate($d);
            $page->meta_description = '';
            $page->intro = $faker->text();
        }

        return $pages->map(function (Page $page) {
            return [$page];
        });
    }

    /**
     * @param Page $page
     *
     * @dataProvider pageProvider
     */
    public function test_update_meta_description_correctly(Page $page)
    {
        $page->changeMetaDescription();
        static::assertTrue(strlen($page->meta_description) <= strlen($page->intro_text));
    }


    /**
     * @param Page $page
     *
     * @dataProvider pageProvider
     */
    public function test_update_meta_description_correctly_even_without_ponctuation(Page $page)
    {
        $page->intro = preg_replace('/[.,?]/', '', $page->intro);
        $page->changeMetaDescription();
        static::assertTrue(strlen($page->meta_description) <= strlen($page->intro_text));
    }
}
