<?php

/**
 * Created by IntelliJ IDEA.
 * User: loic
 * Date: 05/10/2020
 * Time: 14:47
 */

namespace Bloom\CMS\Core;

use Bloom\CMS\Core\Contenus\PageStatique;
use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Framework\Admin\AdminMenu;
use Bloom\CMS\Core\Framework\Module;
use Bloom\CMS\Core\Helpers\Workflow;
use Bloom\CMS\Core\Helpers\WorkflowsFactory;
use Bloom\CMS\Core\Http\Controllers\AdminController;
use Bloom\CMS\Core\Http\Dossier;
use Bloom\CMS\Core\Http\Page;
use Closure;
use Illuminate\Console\Command;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Str;
use InvalidArgumentException;
use ReflectionClass;

/**
 * Class BloomProvider
 *
 * @package Bloom\CMS\Core
 *
 * @property Module|null $module Instance du module en bdd
 */
class BloomProvider extends ServiceProvider
{
    /**
     * Nom public du module
     *
     * @var string|null
     */
    protected $nom = null;
    /**
     * Code du module ( nom privé )
     *
     * @var string|null
     */
    protected $code = null;
    /**
     * Le module doit-il remonté dans l'admin? Aka doit-il avoir un menu
     *
     * @var bool
     */
    protected $has_admin = true;
    /**
     * Icons du menu dans l'admin
     *
     * @var string
     */
    protected $icons = "paw";
    /**
     * @var Command[]
     */
    protected $registeredCommands = [];

    /**
     * Le namespace Twig du module ( est valorisé par ucfirst($code) )
     *
     * @var string
     */
    private $twig_namespace;
    /**
     * Le nom "vendor" du module
     *
     * @var string
     */
    private $vendor;
    /**
     * Le dossier racine du module
     *
     * @var string
     */
    private $dir;
    /**
     * Le nom de la route index ( valorisé par admin_$code_index )
     *
     * @var string
     */
    private $index = '';

    /**
     * BloomProvider constructor.
     *
     * @param Application $app
     *
     * @throws InvalidArgumentException
     */
    public function __construct(Application $app)
    {
        // Si le nom n'est pas valorisé on leve l'exception
        if ($this->nom === null) {
            throw new InvalidArgumentException("[" . static::class . "] 'nom' ne peut pas être null");
        }

        // Si le code n'est pas valorisé on leve l'exception
        if ($this->code === null) {
            throw new InvalidArgumentException("[" . static::class . "] 'code' ne peut pas être null");
        }

        $this->twig_namespace = ucfirst($this->code);
        $this->vendor = "cms-module-" . $this->code;
        if ($this->has_admin) {
            $this->index = "admin_" . $this->code . "_index";
        }
        $file = (new ReflectionClass($this))->getFileName();
        $this->dir = dirname(dirname($file));
        if ($this->dir === dirname(__DIR__) && $this->code !== 'core') {
            // Si le dossier est le parent de BloomProvider c'est qu'il y'a un probleme
            throw new InvalidArgumentException('Impossible de définir le dossier du module ' . static::class);
        }
        parent::__construct($app);
    }

    public function register()
    {
        if ($this->app->runningInConsole()) {
            $this->commands($this->registeredCommands);
        }
        $this->loadFactoriesFrom($this->dir . '/database/factories');
        $this->loadMigrationsFrom($this->dir . '/database/migrations');
        $this->loadViewsFrom($this->dir . '/resources/views', $this->twig_namespace);
        $this->registerRouteFromFiles();

        if (is_dir($this->dir . '/public') && count(scandir($this->dir . '/public')) > 0) {
            // On mets à dispo les fichiers dans le groups "assets" uniquement si il y'a des fichiers dans "public"
            $this->publishes(
                [
                    $this->dir . '/public' => public_path('/vendor/' . $this->vendor)
                ],
                'assets'
            );
        }
        if (is_dir($this->dir . '/config')) {
            foreach (scandir($this->dir . '/config') as $file) {
                if (in_array($file, ['.', '..'])) {
                    continue;
                }
                $this->mergeConfigFrom($this->dir . '/config/' . $file, str_replace('.php', '', $file));
            }
        }
    }

    public function boot()
    {
        $this->getModule();
    }

    /**
     * @return string
     */
    public function getDir(): string
    {
        return $this->dir;
    }

    public function getCoreDir(): string
    {
        return dirname(__DIR__);
    }

    public function getCode(): string
    {
        return $this->code;
    }

    /**
     * Ajout d'un sous menu dans l'admin
     *
     * @param AdminMenu[] $menus
     */
    protected function addSubMenus(array $menus): void
    {
        $self = $this;
        $closure = function () use ($menus, $self) {
            if (!Schema::hasTable('admin_menus')) {
                // Si pas de table, sa veut dire qu'on est à l'install du projet
                return;
            }
            if (AdminMenu::query()->where('module_id', '=', $self->module->id)->exists()) {
                // Si les admin menu pour le module existe déjà on skip
                return;
            }
            foreach ($menus as $menu) {
                $menu->module_id = $self->module->id;
                $menu->active = true;
                $menu->save();
            }
        };
        if ($this->app->resolved('db')) {
            // Juste pour être sur on vérifie que la base de donnée est prête
            $closure();
        } else {
            $this->app->afterResolving('db', $closure);
        }
    }

    /**
     * @param WorkflowsFactory $factory
     */
    protected function haveGenericIndex(WorkflowsFactory $factory)
    {
        // On register la route admin_$code_index
        $this->registerRoute(
            function () {
                Route::get('/{worflow:code?}', AdminController::class . '@listing')->name('index');
            },
            true,
            false
        );

        // On sauvegarde les workflows si il n'existe pas
        if (Schema::hasTable('workflows') && Workflow::is_module($this->module)->count() === 0) {
            $factory->saveFor($this->module);
        }
    }

    /**
     * Enregistre un dossier avec une page statique en index, doit être appelé après ou pendant le boot
     *
     * @param string      $type       Type de contenu présent dans le dossier
     *                                (aliasé ou non; ex: articles OU Bloom\Contenu\Articles)
     * @param string      $display    Nom du contenu (sert aussi pour le nom du dossier ainsi que le titre de l'index)
     * @param string      $controller Controller front qui gerera l'index front
     *
     * @param string|null $action     Texte a affiché dans l'archi (action personalisé), par défaut "Gérer les $display"
     * @param string|null $route      Lien du button d'action, par défaut la route index admin
     *
     * @return void
     */
    protected function registerDossier(
        string $type,
        string $display,
        string $controller,
        string $action = null,
        string $route = null
    ) {
        $action = $action ?? 'Gérer les ' . $display;
        $route = $route ?? $this->index;
        if (
            Schema::hasTable('dossiers') &&
            !Dossier::query()->where('type', '=', $type)->exists()
        ) {
            $statique = new PageStatique(
                [
                    'display_type' => $display,
                    'action'       => $action,
                    'route'        => $route,
                    'controller'   => $controller
                ]
            );
            $statique->save();
            // Création du dossier avec un nom filtrable pour le rattachement des pages enfants
            $dossier = new Dossier(
                [
                    'type' => $type,
                    'name' => $display,
                    'slug' => Str::slug($display)
                ]
            );
            $dossier->save();
            // Création de la page index
            $page = new Page();
            $page->contenu()->associate($statique);
            $page->dossier()->associate($dossier);
            $page->titre = $display;
            $page->slug = "";
            $page->meta_description = "";
            $page->meta_titre = "";
            $page->intro = "";
            $page->multiplicity = 1;
            $page->statut_id = Statut::PUBLIE;
            $page->is_index = 1;
            $page->save();
        }
    }

    /**
     * @param callable $route
     * @param bool     $admin          Ajout de régle dédié a l'admin (prefix de nom, middleware auth ..) [default:
     *                                 false]
     * @param bool     $localNamespace Génére dynamiquement le prefix de namespace des controllers [default: true]
     *
     * @SuppressWarnings(PHPMD)
     */
    protected function registerRoute(callable $route, bool $admin = false, bool $localNamespace = true): void
    {
        $self = $this;
        $rules = [
            'middleware' => [
                'web',
                function ($request, Closure $next) use ($self) {
                    // Le middleware est appelé uniquement lors de la résolution de la request
                    // la base de donnée sera donc correctement instancié
                    $request->module_actif = $self->module->id;

                    return $next($request);
                }
            ],
        ];
        if ($localNamespace) {
            $rules['namespace'] = str_replace(class_basename(static::class), '', static::class) . 'Http\Controllers';
        }
        if ($admin) {
            $rules['middleware'][] = 'auth';
            $rules['prefix'] = config('app.admin') . '/' . $this->code;
            Route::name("admin_" . $this->code . '_')->group(
                function () use ($route, $rules) {
                    Route::group(
                        $rules,
                        function () use ($route) {
                            $route();
                        }
                    );
                }
            );
        } else {
            Route::group(
                $rules,
                function () use ($route) {
                    $route();
                }
            );
        }
    }

    /**
     * Register les routes déclarer dans le dossier "routes" du module
     */
    protected function registerRouteFromFiles()
    {
        $dirname = $this->dir . '/routes';
        if (!is_dir($dirname)) {
            return;
        }
        foreach (scandir($dirname) as $file) {
            if (!in_array($file, ['.', '..'])) {
                $this->registerRoute(
                    function () use ($dirname, $file) {
                        $this->loadRoutesFrom($dirname . '/' . $file);
                    },
                    $file === "admin.php"
                );
            }
        }
    }

    /**
     * Return current module
     *
     * @return Module|null
     */
    protected function getModule(): ?Module
    {
        static $module;
        if ($module !== null) {
            return $module;
        }
        if (Schema::hasTable('modules')) {
            /**
             * @var Module $module
             */
            $module = Module::nom($this->code)->first();
            if ($module !== null && !$module->exists) {
                $module->fill(
                    [
                        'nom'        => $this->code,
                        'nom_public' => $this->nom,
                        'route'      => $this->index,
                        'picto'      => $this->icons,
                    ]
                );
                $module->active = 1;
                $module->save();
            } elseif ($module === null) {
                $module = new Module(
                    [
                        'nom'        => $this->code,
                        'nom_public' => $this->nom,
                        'route'      => $this->index,
                        'picto'      => $this->icons,
                        'active'     => 1
                    ]
                );
                $module->save();
            }

            return $module;
        }

        return null;
    }

    /**
     * @param string $name
     *
     * @return ?Module
     */
    public function __get(string $name): ?Module
    {
        switch ($name) {
            case 'module':
                return $this->getModule();
        }

        return null;
    }
}
