<?php

namespace Bloom\CMS\Core\Http;

use Bloom\CMS\Core\Contenus\Applicabilite;
use Bloom\CMS\Core\Contenus\Contenu;
use Bloom\CMS\Core\Contenus\HasContenu;
use Bloom\CMS\Core\Contenus\HasStatut;
use Bloom\CMS\Core\Contenus\IsUserTracked;
use Bloom\CMS\Core\Contenus\PageStatique;
use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Http\Indexes\Section;
use Bloom\CMS\Core\Http\Redirect\Redirection;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Auth\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use InvalidArgumentException;
use Bloom\Cms\Sites\Models\Site;

/**
 * Class Page
 * @package Bloom\CMS\Core\Http
 *
 *          Natural
 *
 * @property string              $titre
 * @property string              $meta_titre
 * @property string              $meta_description
 * @property string              $slug
 * @property string              $pathname
 * @property string              $intro
 * @property string              $multiplicity
 * @property string              $image_opengraph
 * @property int                 $statut_id
 * @property bool                $is_index
 * @property bool                $noindex
 * @property int                 $dossier_id
 * @property int                 $id
 * @property string              $contenu_type
 * @property string              $mentions
 * @property int                 $created_by
 * @property int                 $last_modified_by
 *
 *           Computed
 *
 * @property Dossier             $dossier
 * @property Statut              $statut
 * @property User                $cree_par
 * @property User                $deriere_modification_par Utilisateur qui a fait la derniere modification de contenu
 * @property Carbon              $to_publish_at
 * @property Carbon              $to_unpublish_at
 * @property string              $intro_text
 * @property Contenu             $contenu
 * @property bool                $is_published
 * @property bool                $have_published_children
 * @property bool                $has_menu
 * @property bool                $has_miroirs
 * @property Miroir[]|Collection $miroirs
 * @property string              $edit_route               Lien vers la page d'édition du contenu
 *
 * @method static Builder|static published()
 * @method static Builder|static archived()
 */
class Page extends Model
{
    use HasContenu;
    use HasStatut;
    use IsUserTracked;

    public const SINGLE = '1';
    public const ALL = '*';
    public const MUTLI = '1-n';

    protected $guarded = [];

    protected $dates = [
        'to_publish_at',
        'to_unpublish_at',
    ];

    public function updatePathname()
    {
        $this->pathname = '';
        if ($this->dossier) {
            $this->pathname = '/' . $this->dossier->full_pathname;
        }
        if ($this->slug) {
            $this->pathname .= '/' . $this->slug;
        }

        $query = $this->newQuery()->where('pathname', '=', $this->pathname);
        if ($this->exists) {
            $query->where('id', '!=', $this->getKey());
        }
        Log::debug('Checking duplicate pathname', ["pathname" => $this->pathname]);
        if (
            // Si on a un doublons d'url
            $query->count() > 0 ||
            // Ou si une redirection en 301 existe déjà sur l'url actuel
            // Cas d'un déplacement de page, le pathname d'origine n'existe plus dans la table page
            // mais il ne faut quand même pas crée de page portant l'ancien pathname
            Redirection::query()
                ->where('statut_id', '=', Statut::PUBLIE)
                ->where('source', '=', $this->pathname)
                ->where('hard', '=', 1)
                ->exists()
        ) {
            $this->uniquePathname($this->pathname);
        }
    }

    protected function uniquePathname(string $original)
    {
        $query = $this->newQuery()->whereRaw('pathname REGEXP "^' . $original . '(-[1-9]+)*$"');
        if ($this->exists) {
            $query->where('id', '!=', $this->getKey());
        }
        $query2 = Redirection::query()
            ->where('statut_id', '=', Statut::PUBLIE)
            ->where('source', '=', $this->pathname)
            ->where('hard', '=', 1);

        $query3 = Miroir::query()->where('pathname', '=', $this->pathname);

        $this->pathname .= "-" . ($query->count() + $query2->count() + $query3->count());
        $this->slug .= "-" . ($query->count() + $query2->count() + $query3->count());
        Log::debug('Setting unique pathname', ["pathname" => $this->pathname]);
    }

    public function changeMetaDescription()
    {
        // Position de la première ponctuation (‘.’, ‘!’ ou ‘?’) dans l'intro trouvée après 100 caractères
        $search_char = ['.', '!', '?'];
        $pos = false;
        $array_pos = [];
        $intro_decode = html_entity_decode($this->intro_text, ENT_QUOTES, 'UTF-8');

        if (Str::length($intro_decode) >= 100) {
            foreach ($search_char as $char) {
                $pos = strpos($intro_decode, $char, 100);
                if (is_numeric($pos)) {
                    array_push($array_pos, $pos <= 180 ? $pos : 180);
                }
            }

            if (count($array_pos) === 0) {
                // On a pas trouvé de ponctuation, donc on cherche le premier espace
                $pos = strpos($intro_decode, ' ', 100);
                if (is_numeric($pos)) {
                    array_push($array_pos, $pos <= 180 ? $pos : 180);
                } else {
                    // Si on a toujours pas trouvé on tronc avant les 100 char
                    $this->meta_description = Str::beforeLast(substr($intro_decode, 0, 90), ' ') . ' [...]';
                    return;
                }
            }

            $pos = min($array_pos);
        }

        // Modification du champ meta_description
        if (!$pos) {
            $this->meta_description = $intro_decode;
        } else {
            $end = $pos >= Str::length($intro_decode) - 6 ? '' : ' [...]';
            $this->meta_description = substr($intro_decode, 0, $pos + 1) . $end;
        }
    }

    public function setMultiplicityAttribute(string $value)
    {
        $accepted = [static::SINGLE, static::MUTLI, static::ALL];
        if (!in_array($value, $accepted)) {
            throw new InvalidArgumentException('multiplicity should be in ' . implode(',', $accepted));
        }

        $this->attributes['multiplicity'] = $value;
    }

    public function setSlugAttribute(string $slug)
    {
        $this->attributes['slug'] = Str::slug(str_replace('/', '', $slug));
    }

    public function miroirs()
    {
        return $this->hasMany(Miroir::class);
    }

    public function miroirNotArchived()
    {
        return $this->hasMany(Miroir::class)->where('statut_id', '!=', Statut::ARCHIVE);
    }

    public function dossier()
    {
        return $this->belongsTo(Dossier::class);
    }

    public function archive(): bool
    {
        $this->statut_id = Statut::ARCHIVE;

        return $this->save();
    }

    public function unarchive(): bool
    {
        $this->statut_id = Statut::EN_COURS;
        $this->to_publish_at = null;
        $this->to_unpublish_at = null;

        return $this->save();
    }

    public function publish(): bool
    {
        $this->to_publish_at = now();
        $this->statut_id = Statut::PUBLIE;

        return $this->save();
    }

    public function softDelete(): bool
    {
        $this->statut_id = Statut::SUPPRIMER;

        return $this->save();
    }

    public function unpublish(): bool
    {
        $this->statut_id = Statut::EN_COURS;
        $this->to_publish_at = null;
        $this->to_unpublish_at = null;

        return $this->save();
    }

    public function getIntroTextAttribute()
    {
        if ((string)$this->intro === '') {
            return '';
        }

        return htmlspecialchars_decode(strip_tags($this->intro), ENT_QUOTES);
    }

    public function getHasMenuAttribute()
    {
        return DB::table('menus')
                ->where('page_id', '=', $this->id)
                ->where('statut_id', '=', 1)
                ->count() > 0;
    }

    public function getMenuAttribute()
    {
        if ($this->has_menu) {
            return DB::table('menus')
                ->where('page_id', '=', $this->id)
                ->where('statut_id', '=', 1)
                ->first();
        }

        return null;
    }

    public function getHasMiroirsAttribute()
    {
        return DB::table('page_dossiers')->where('page_id', '=', $this->id)->exists();
    }

    public function getEditRouteAttribute()
    {
        //#region Deprecié, ces exceptions on été faites en attendant l'ajout des pages statiques
        $exception = ['contact', 'plan_du_site', 'recrutement'];

        if (in_array($this->contenu_type, $exception)) {
            return '#';
        }
        //#endregion
        /**
         * @var Applicabilite[] $applicabilites
         */
        static $applicabilites;
        if ($applicabilites === null || app()->environment('test', 'testing')) {
            $applicabilites = Applicabilite::query()
                ->whereRaw(sprintf('b_on & %d = %d', Applicabilite::ARCHI, Applicabilite::ARCHI))->get();
        }
        foreach ($applicabilites as $applicabilite) {
            if ($this->contenu instanceof $applicabilite->contenu_type) {
                return route($applicabilite->route, $this->contenu);
            }
        }

        if ($this->is_index) {
            return 'document.dispatchEvent(new CustomEvent("edit-sommaire", {detail: {
                title: "Modifier le sommaire ' .
                (Section::query()->where('page_id', '=', $this->id)->count() > 0 ? 'personnalisé' : 'automatique') .
                '",
                route: "' .
                route(
                    'admin_archi_som',
                    [
                        Section::query()->where('page_id', '=', $this->id)->count() > 0 ? 'perso' : 'auto',
                        $this
                    ]
                ) . '"}}))';
        }

        return '#';
    }

    public function scopeType(Builder $query, string $type)
    {
        return $query->where('contenu_type', '=', $type);
    }

    public function getHavePublishedChildrenAttribute(): bool
    {
        if (!$this->is_index) {
            return false;
        }

        foreach ($this->dossier->pages as $page) {
            if ($page->id == $this->id) {
                continue;
            }

            if ($page->is_published) {
                return true;
            }
        }

        return false;
    }

    public function getContenuNameAttribute(): string
    {
        if ($this->contenu && $this->contenu instanceof PageStatique) {
            return $this->contenu->display_type;
        }

        return Str::title($this->contenu_type);
    }

    public function sites()
    {
        return $this->belongsToMany(Site::class, 'site_pages', 'pages_id', 'sites_id')->withPivot(['created_at', 'updated_at']);
    }

    static public function getPagesWithSites()
    {
        $pages = Page::with('sites')
            ->where('statut_id', '=', Statut::PUBLIE)
            ->get();

        return $pages;
    }
}
