<?php

namespace Bloom\CMS\Core\Http;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Http\Redirect\Redirection;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Auth\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

/**
 * Class Page
 *
 * @package Bloom\CMS\Modules\Architecture\Model
 *
 *          Natural
 *
 * @property int     $id
 * @property int     $statut_id
 * @property string  $pathname
 * @property int     dossier_id
 *
 *
 *           Computed
 *
 * @property Page    $page
 * @property Dossier $dossier
 * @property Statut  $statut
 * @property User    $cree_par
 * @property User    $deriere_modification_par
 * @property bool    is_published
 * @property bool    have_published_children
 * @property bool    has_menu
 *
 * @method static Builder|static published()
 * @method static Builder|static archived()
 */
class Miroir extends Model
{
    protected $table = 'page_dossiers';

    protected $appends = ['slug'];


    public static function booted()
    {
        static::saving(
            function (Miroir $page) {
                // Si le "dossier" est mis à jour, on met à jour le pathname
                $page->updatePathname();
            }
        );
    }

    public function updatePathname()
    {
        $this->pathname = '';
        if ($this->dossier) {
            $this->pathname = '/' . $this->dossier->full_pathname;
        }
        if ($this->page->slug) {
            $this->pathname .= '/' . $this->page->slug;
        }

        $query = $this->newQuery()->where('pathname', '=', $this->pathname);
        if ($this->exists) {
            $query->where('id', '!=', $this->getKey());
        }

        Log::debug('Checking duplicate pathname', ["pathname" => $this->pathname]);
        if (
            // Si on a un doublons d'url
            $query->count() > 0
            // Ou si une redirection en 301 existe déjà sur l'url actuel
            // Cas d'un déplacement de page, le pathname d'origine n'existe plus dans la table page
            // mais il ne faut quand même pas crée de page portant l'ancien pathname
            || Redirection::query()
                ->where('statut_id', '=', Statut::PUBLIE)
                ->where('source', '=', $this->pathname)
                ->where('hard', '=', 1)
                ->exists()
            || Page::query()->where('pathname', '=', $this->pathname)->exists()
        ) {
            $this->uniquePathname($this->pathname);
        }
    }

    protected function uniquePathname(string $original)
    {
        $query = $this->newQuery()->whereRaw('pathname REGEXP "^' . $original . '(-[1-9]+)*$"');
        if ($this->exists) {
            $query->where('id', '!=', $this->getKey());
        }
        $query2 = Redirection::query()
            ->where('statut_id', '=', Statut::PUBLIE)
            ->where('source', '=', $this->pathname)
            ->where('hard', '=', 1);

        $query3 = Page::query()->whereRaw('pathname REGEXP "^' . $original . '(-[1-9]+)*$"');

        $this->pathname .= "-" . ($query->count() + $query2->count() + $query3->count());
        Log::debug('Setting unique pathname', ["pathname" => $this->pathname]);
    }

    public function getSlugAttribute()
    {
        preg_match_all('~.*/([^/]+)$~m', $this->pathname, $matches);

        return $matches[1];
    }


    public function page()
    {
        return $this->belongsTo(Page::class);
    }

    public function dossier()
    {
        return $this->belongsTo(Dossier::class);
    }

    public function statut()
    {
        return $this->belongsTo(Statut::class);
    }

    //phpcs:disable

    /** @noinspection PhpMethodNamingConventionInspection */
    public function cree_par()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /** @noinspection PhpMethodNamingConventionInspection */
    public function deriere_modification_par()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    //phpcs:enable

    public function scopePublished(Builder $query)
    {
        return $query->where('statut_id', '=', Statut::PUBLIE);
    }

    public function scopeArchived(Builder $query)
    {
        return $query->where('statut_id', '=', Statut::ARCHIVE);
    }

    public function archive(): bool
    {
        $this->statut_id = Statut::ARCHIVE;

        return $this->save();
    }

    public function unarchive(): bool
    {
        $this->statut_id = Statut::EN_COURS;

        return $this->save();
    }

    public function publish(): bool
    {
        $this->statut_id = Statut::PUBLIE;

        return $this->save();
    }

    public function unpublish(): bool
    {
        $this->statut_id = Statut::EN_COURS;

        return $this->save();
    }

    public function softDelete(): bool
    {
        $this->statut_id = Statut::SUPPRIMER;

        return $this->save();
    }

    public function getIsPublishedAttribute(): bool
    {
        return $this->statut_id == Statut::PUBLIE;
    }

    public function getHasMenuAttribute()
    {
        return DB::table('menus')
                ->where('miroir_id', '=', $this->id)
                ->where('statut_id', '=', 1)
                ->count() > 0;
    }

    public function getMenuAttribute()
    {
        if ($this->has_menu) {
            return DB::table('menus')
                ->where('miroir_id', '=', $this->id)
                ->where('statut_id', '=', 1)
                ->first();
        }

        return null;
    }
}
