<?php
/**
 * Created by IntelliJ IDEA.
 * User: loic
 * Date: 26/11/2020
 * Time: 15:08
 */

namespace Feature;


use Bloom\CMS\Core\CmsCoreProvider;
use Bloom\CMS\Core\Contenus\PageStatique;
use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Http\Dossier;
use Bloom\CMS\Core\Http\Events\PageArchived;
use Bloom\CMS\Core\Http\Events\PageDeleted;
use Bloom\CMS\Core\Http\Events\PagePublished;
use Bloom\CMS\Core\Http\Events\PageStatutChange;
use Bloom\CMS\Core\Http\Events\PageUrlChange;
use Bloom\CMS\Core\Http\Indexes\Section;
use Bloom\CMS\Core\Http\Indexes\SectionLien;
use Bloom\CMS\Core\Http\Miroir;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\Core\Http\Redirect\Redirection;
use Bloom\CMS\Core\Test;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Foundation\Auth\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Event;

class PageTest extends Test
{
    protected $provider = CmsCoreProvider::class;

    protected function setUp(): void
    {
        parent::setUp();
        // On a besoin de s'authentifié car Page fixe lui même le created_by et last_modified_by
        Auth::setUser(User::first());
    }


    public function provider(): Collection
    {
        $collections = new Collection();

        for ($i = 0; $i < 5; $i++) {
            $d = factory(Dossier::class)->create();
            $page = factory(Page::class)
                ->state('index')
                ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);
            $collections[] = $page;
        }

        return $collections;
    }

    public function nonPublishedProvider(): Collection
    {
        $collections = new Collection();

        for ($i = 0; $i < 5; $i++) {
            $d = factory(Dossier::class)->create();
            $page = factory(Page::class)
                ->state('index')
                ->state('unpublished')
                ->create(['dossier_id' => $d->id]);
            $collections[] = $page;
        }

        return $collections;
    }

    public function test_page_should_return_200_if_published()
    {
        $pages = $this->provider();
        CmsCoreProvider::refreshRoutes();

        foreach ($pages as $page) {
            $response = $this->get($page->pathname);
            // La page est publié sans redirection donc le statut doit être 200
            $response->assertOk();
        }
    }

    public function test_page_should_return_404_if_not_published()
    {
        $pages = $this->nonPublishedProvider();
        CmsCoreProvider::refreshRoutes();

        foreach ($pages as $page) {
            $response = $this->get($page->pathname);
            // Les pages n'étant pas publié et n'ayant jamais été publié on est sensé avoir une 404
            $response->assertNotFound();
        }
    }

    public function test_page_should_redirect_to_parent_if_unpublished()
    {
        $pages = $this->provider();
        $pages->each(function (Page $page) {
            $page->unpublish();
        });

        CmsCoreProvider::refreshRoutes();

        foreach ($pages as $page) {
            $response = $this->get($page->pathname);
            // Toutes les pages testé sont des index de dossier de premier niveau
            // Le dossier parent est donc la racine, on test donc la redirection sur "/"
            $response->assertRedirect('/');
            // Les redirection de dépublication sont en statut 302, redirection temporaire
            $response->assertStatus(302);
        }
    }

    public function test_page_should_redirect_301_if_pathname_changed()
    {
        $pages = $this->provider();
        $old = $pages->pluck('pathname');
        $pages->each(function (Page $page) {
            $page->slug = 'sommaire';
            $page->save();
        });

        CmsCoreProvider::refreshRoutes();

        foreach ($pages as $k => $page) {
            $response = $this->get($old[ $k ]);
            $response->assertRedirect($page->pathname);
            // Les redirection de changement de pathname sont en 301, redirect permantant
            $response->assertStatus(301);
        }
    }

    public function test_page_should_have_unique_pathname()
    {
        $d = factory(Dossier::class)->create();
        /**
         * @var Page $page
         * @var Page $page2
         */
        $page = factory(Page::class)
            ->state('index')
            ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);

        $page2 = factory(Page::class)->make(['pathname' => $page->pathname, 'slug' => $page->dossier->slug]);

        // On vérifie que les deux sont bien éguaux avant d'enregistré
        static::assertEquals($page->pathname, $page2->pathname);

        $page2->save();
        static::assertNotEquals($page->pathname, $page2->pathname);
        static::assertEquals($page->pathname . '-1', $page2->pathname);
    }

    /**
     * @depends test_page_should_have_unique_pathname
     */
    public function test_page_should_have_unique_pathname_multiple()
    {

        $d = factory(Dossier::class)->create();
        /**
         * @var Page $page
         * @var Page $page2
         * @var Page $page3
         */
        $page = factory(Page::class)
            ->state('index')
            ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);

        $page2 = factory(Page::class)->make(['pathname' => $page->pathname, 'slug' => $page->dossier->slug]);
        $page3 = factory(Page::class)->make(['pathname' => $page->pathname, 'slug' => $page->dossier->slug]);
        $page2->save();
        $page3->save();
        // La 3iem page doit correctement comprendre qu'il y'a déjà un doublons pour le pathname et correctement
        // valorisé le chiffre en fin d'url
        static::assertEquals($page->pathname . '-2', $page3->pathname);
    }

    public function test_page_should_not_trigger_event_if_does_not_exist_yet()
    {
        Event::fake([PageStatutChange::class, PageUrlChange::class, PagePublished::class]);

        factory(Page::class)->create();

        Event::assertNotDispatched(PageUrlChange::class);
        Event::assertNotDispatched(PageStatutChange::class);
        Event::assertNotDispatched(PagePublished::class);
    }

    public function test_page_should_trigger_event_if_exist()
    {
        Event::fake([PageStatutChange::class, PagePublished::class]);

        /**
         * @var Page $page
         */
        // On crée une page statiques pour l'occassion
        $contenu = new PageStatique(['controller' => '', 'display_type' => '', 'action' => '', 'route' => '']);
        $contenu->save();
        $page = factory(Page::class)
            ->state('unpublished')
            ->create(['contenu_type' => 'page_statiques', 'contenu_id' => $contenu->id]);

        $page->publish();

        Event::assertDispatched(PageStatutChange::class);
        Event::assertDispatched(PagePublished::class);
    }

    public function test_page_should_trigger_event_if_deleted()
    {
        Event::fake([PageStatutChange::class, PageDeleted::class]);

        /**
         * @var Page $page
         */
        // On crée une page statiques pour l'occassion
        $contenu = new PageStatique(['controller' => '', 'display_type' => '', 'action' => '', 'route' => '']);
        $contenu->save();
        $page = factory(Page::class)
            ->state('unpublished')
            ->create(['contenu_type' => 'page_statiques', 'contenu_id' => $contenu->id]);

        $page->softDelete();

        Event::assertDispatched(PageStatutChange::class);
        Event::assertDispatched(PageDeleted::class);
    }

    public function test_page_should_trigger_event_if_pathname_change()
    {
        Event::fake([PageUrlChange::class]);

        /**
         * @var Page $page
         */
        // On crée une page statiques pour l'occassion
        $contenu = new PageStatique(['controller' => '', 'display_type' => '', 'action' => '', 'route' => '']);
        $contenu->save();
        $page = factory(Page::class)
            ->state('unpublished')
            ->create(['contenu_type' => 'page_statiques', 'contenu_id' => $contenu->id]);

        $page->slug = '/test';
        $page->save();

        Event::assertDispatched(PageUrlChange::class);
    }

    public function test_page_should_trigger_event_archived()
    {
        Event::fake([PageStatutChange::class, PageArchived::class]);

        /**
         * @var Page $page
         */
        // On crée une page statiques pour l'occassion
        $contenu = new PageStatique(['controller' => '', 'display_type' => '', 'action' => '', 'route' => '']);
        $contenu->save();
        $page = factory(Page::class)
            ->state('unpublished')
            ->create(['contenu_type' => 'page_statiques', 'contenu_id' => $contenu->id]);

        $page->archive();

        Event::assertDispatched(PageStatutChange::class);
        Event::assertDispatched(PageArchived::class);
    }

    public function test_moroir_should_follow_the_canonical_statut()
    {
        /**
         * @var Page $page
         */
        // On crée une page statiques pour l'occassion
        $contenu = new PageStatique(['controller' => '', 'display_type' => '', 'action' => '', 'route' => '']);
        $contenu->save();
        $page = factory(Page::class)
            ->state('unpublished')
            ->create(['contenu_type' => 'page_statiques', 'contenu_id' => $contenu->id]);
        $d = factory(Dossier::class)->create();
        $miroir = new Miroir();
        $miroir->page()->associate($page);
        $miroir->dossier()->associate($d);
        $miroir->statut_id = 1;
        $miroir->cree_par()->associate(Auth::user());
        $miroir->deriere_modification_par()->associate(Auth::user());
        $miroir->save();

        foreach ([Statut::ARCHIVE, Statut::SUPPRIMER, Statut::PUBLIE, Statut::EN_COURS] as $statut) {
            $page->statut_id = $statut;
            $page->save();
            $miroir->refresh();

            static::assertEquals($statut, (int)$miroir->statut_id);
        }
    }

    public function test_should_followed_up_the_redirection()
    {
        $d = factory(Dossier::class)->create();
        /**
         * @var Page $page
         * @var Page $page2
         */
        $page = factory(Page::class)
            ->state('index')
            ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);

        $page->slug = 'index';
        $page->save();
        $page->slug = 'listing';
        $page->save();

        CmsCoreProvider::refreshRoutes();

        static::assertEquals(2, Redirection::query()->count());

        $response = $this->get($page->dossier->slug);
        $response->assertRedirect($page->pathname);
        $response->assertStatus(301);
    }

    public function test_access_to_miroir_should_display_the_canonical()
    {

        $d = factory(Dossier::class)->create();
        $d2 = factory(Dossier::class)->create();
        /**
         * @var Page $page
         * @var Page $page2
         */
        $page = factory(Page::class)
            ->state('index')
            ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);

        $miroir = new Miroir();
        $miroir->page()->associate($page);
        $miroir->dossier()->associate($d2);
        $miroir->statut_id = 1;
        $miroir->cree_par()->associate(Auth::user());
        $miroir->deriere_modification_par()->associate(Auth::user());
        $miroir->save();

        CmsCoreProvider::refreshRoutes();

        $response = $this->get($miroir->pathname);
        $response->assertOk();
        $response->assertSessionHas('canonical_link', $page->pathname);
        $response->assertViewIs('Archi::front.index_auto');
    }

    public function test_index_auto_should_load_section_ordered()
    {
        $d = factory(Dossier::class)->create();
        /**
         * @var Page $page
         * @var Page $page2
         */
        $page = factory(Page::class)
            ->state('index')
            ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);

        $contenu = new PageStatique(['controller' => '', 'display_type' => '', 'action' => '', 'route' => '']);
        $contenu->save();
        $p = factory(Page::class)->create(['contenu_type' => 'page_statiques', 'contenu_id' =>$contenu->id]);

        $section = factory(Section::class)->make(['rang' => 2]);
        $section->page_id = $page->id;
        $section->save();
        factory(SectionLien::class)->create([
            'section_id' => $section->id,
            'page_id' => $p->id,
        ]);
        $section2 = factory(Section::class)->make(['rang' => 1]);
        $section2->page_id = $page->id;
        $section2->save();
        factory(SectionLien::class)->create([
            'section_id' => $section2->id,
            'page_id' => $p->id,
        ]);

        CmsCoreProvider::refreshRoutes();

        $response = $this->get($page->pathname);
        $response->assertOk();
        $response->assertViewIs('Archi::front.index_perso');
        $sections = $response->viewData('sections');
        static::assertInstanceOf(Collection::class, $sections);
        foreach ($sections as $k=> $section) {
            static::assertEquals($k+1, (int)$section->rang);
        }
    }

    public function test_slug_modif_on_directory_should_modify_pathname_of_all_the_pages_in_dir()
    {
        Event::fake([PageUrlChange::class]);
        $d = factory(Dossier::class)->create();
        $collections = new Collection();
        for ($i = 0; $i < 5; $i++) {
            $page = factory(Page::class)
                ->state('index')
                ->create(['dossier_id' => $d->id, 'pathname' => '/' . $d->slug]);
            $collections[] = $page;
        }

        $d->slug = 'test';
        $d->save();

        $collections->each(function(Page $page) {
            $page->refresh();
        });

        foreach ($collections as $page) {
            Event::assertDispatched(PageUrlChange::class, function(PageUrlChange $event) use($page) {
                return $event->page->id == $page->id;
            });
            static::assertStringStartsWith('/test', $page->pathname);
        }
    }
}
