<?php

/**
 * Created by IntelliJ IDEA.
 * User: loic
 * Date: 05/10/2020
 * Time: 14:47
 */

namespace Bloom\CMS\Core;

use Bloom\CMS\Core\Framework\Admin\AdminMenu;
use Bloom\CMS\Core\Framework\Module;
use Closure;
use Illuminate\Console\Command;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\ServiceProvider;
use InvalidArgumentException;
use ReflectionClass;
use ReflectionException;

/**
 * Class BloomProvider
 *
 * @package Bloom\CMS\Core
 *
 * @property Module|null $module Instance du module en bdd
 */
class BloomProvider extends ServiceProvider
{
    /**
     * Nom public du module
     *
     * @var string|null
     */
    protected $nom = null;
    /**
     * Code du module ( nom privé )
     *
     * @var string|null
     */
    protected $code = null;
    /**
     * Le module doit-il remonté dans l'admin? Aka doit-il avoir un menu
     *
     * @var bool
     */
    protected $has_admin = true;
    /**
     * Icons du menu dans l'admin
     *
     * @var string
     */
    protected $icons = "paw";
    /**
     * @var Command[]
     */
    protected $registeredCommands = [];

    /**
     * Le namespace Twig du module ( est valorisé par ucfirst($code) )
     *
     * @var string
     */
    private $twig_namespace;
    /**
     * Le nom "vendor" du module
     *
     * @var string
     */
    private $vendor;
    /**
     * Le dossier racine du module
     *
     * @var string
     */
    private $dir;
    /**
     * Le nom de la route index ( valorisé par admin_$code_index )
     *
     * @var string
     */
    private $index = '';

    /**
     * BloomProvider constructor.
     *
     * @param Application $app
     *
     * @throws ReflectionException
     * @throws InvalidArgumentException
     */
    public function __construct(Application $app)
    {
        // Si le nom n'est pas valorisé on leve l'exception
        if ($this->nom === null) {
            throw new InvalidArgumentException("[" . static::class . "] 'nom' ne peut pas être null");
        }

        // Si le code n'est pas valorisé on leve l'exception
        if ($this->code === null) {
            throw new InvalidArgumentException("[" . static::class . "] 'code' ne peut pas être null");
        }

        $this->twig_namespace = ucfirst($this->code);
        $this->vendor = "cms-module-" . $this->code;
        if ($this->has_admin) {
            $this->index = "admin_" . $this->code . "_index";
        }
        $file = (new ReflectionClass($this))->getFileName();
        $this->dir = dirname(dirname($file));
        if ($this->dir === dirname(__DIR__) && $this->code !== 'core') {
            // Si le dossier est le parent de BloomProvider c'est qu'il y'a un probleme
            throw new InvalidArgumentException('Impossible de définir le dossier du module ' . static::class);
        }
        parent::__construct($app);
    }

    public function register()
    {
        if ($this->app->runningInConsole()) {
            $this->commands($this->registeredCommands);
        }
        $this->loadFactoriesFrom($this->dir . '/database/factories');
        $this->loadMigrationsFrom($this->dir . '/database/migrations');
        $this->loadViewsFrom($this->dir . '/resources/views', $this->twig_namespace);
        $this->registerRoute();

        if (is_dir($this->dir . '/public') && count(scandir($this->dir . '/public')) > 0) {
            // On mets à dispo les fichiers dans le groups "assets" uniquement si il y'a des fichiers dans "public"
            $this->publishes(
                [
                    $this->dir . '/public' => public_path('/vendor/' . $this->vendor)
                ],
                'assets'
            );
        }
        if (is_dir($this->dir . '/config')) {
            foreach (scandir($this->dir . '/config') as $file) {
                if (in_array($file, ['.', '..'])) {
                    continue;
                }
                $this->mergeConfigFrom($this->dir . '/config/' . $file, str_replace('.php', '', $file));
            }
        }
    }

    public function boot()
    {
        $this->getModule();
    }

    /**
     * @return string
     */
    public function getDir(): string
    {
        return $this->dir;
    }

    public function getCoreDir(): string
    {
        return dirname(__DIR__);
    }

    public function getCode(): string
    {
        return $this->code;
    }

    /**
     * Ajout d'un sous menu dans l'admin
     *
     * @param AdminMenu[] $menus
     */
    protected function addSubMenus(array $menus): void
    {
        $self = $this;
        $closure = function () use ($menus, $self) {
            if (!Schema::hasTable('admin_menus')) {
                // Si pas de table, sa veut dire qu'on est à l'install du projet
                return;
            }
            if (AdminMenu::query()->where('module_id', '=', $self->module->id)->exists()) {
                // Si les admin menu pour le module existe déjà on skip
                return;
            }
            foreach ($menus as $menu) {
                $menu->module_id = $self->module->id;
                $menu->active = true;
                $menu->save();
            }
        };
        if ($this->app->resolved('db')) {
            // Juste pour être sur on vérifie que la base de donnée est prête
            $closure();
        } else {
            $this->app->afterResolving('db', $closure);
        }
    }

    /**
     * Register les routes déclarer dans le dossier "routes" du module
     *
     * @SuppressWarnings(PHPMD)
     */
    protected function registerRoute(): void
    {
        $dirname = $this->dir . '/routes';
        if (!is_dir($dirname)) {
            return;
        }
        $self = $this;
        foreach (scandir($dirname) as $file) {
            if (!in_array($file, ['.', '..'])) {
                $namespace = str_replace(class_basename(static::class), '', static::class) . 'Http\Controllers';
                $rules = [
                    'middleware' => ['web', function ($request, Closure $next) use ($self) {
                        // Le middleware est appelé uniquement lors de la résolution de la request
                        // la base de donnée sera donc correctement instancié
                        $request->module_actif = $self->module->id;

                        return $next($request);
                    }],
                    'namespace'  => $namespace
                ];
                if ($file === "admin.php") {
                    $rules['middleware'][] = 'auth';
                    $rules['prefix'] = config('app.admin') . '/' . $this->code;
                    Route::name("admin_" . $this->code . '_')->group(
                        function () use ($dirname, $file, $rules) {
                            Route::group(
                                $rules,
                                function () use ($dirname, $file) {
                                    $this->loadRoutesFrom($dirname . '/' . $file);
                                }
                            );
                        }
                    );
                } else {
                    Route::group(
                        $rules,
                        function () use ($dirname, $file) {
                            $this->loadRoutesFrom($dirname . '/' . $file);
                        }
                    );
                }
            }
        }
    }

    /**
     * Return current module
     *
     * @return Module|null
     */
    protected function getModule(): ?Module
    {
        static $module;
        if ($module !== null) {
            return $module;
        }
        if (Schema::hasTable('modules')) {
            /**
             * @var Module $module
             */
            $module = Module::nom($this->code)->first();
            if ($module !== null && !$module->exists) {
                $module->fill(
                    [
                        'nom'        => $this->code,
                        'nom_public' => $this->nom,
                        'route'      => $this->index,
                        'picto'      => $this->icons,
                    ]
                );
                $module->active = 1;
                $module->save();
            } elseif ($module === null) {
                $module = new Module(
                    [
                        'nom'        => $this->code,
                        'nom_public' => $this->nom,
                        'route'      => $this->index,
                        'picto'      => $this->icons,
                        'active'     => 1
                    ]
                );
                $module->save();
            }

            return $module;
        }

        return null;
    }

    /**
     * @param string $name
     *
     * @return mixed
     */
    public function __get(string $name)
    {
        switch ($name) {
            case 'module':
                return $this->getModule();
        }

        return null;
    }
}
