<?php

/**
 * Created by IntelliJ IDEA.
 * User: loic
 * Date: 22/03/2021
 * Time: 15:23
 */

namespace Bloom\CMS\Core\Helpers;

use Bloom\CMS\Core\Framework\Module;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * Class Workflow
 * @package Bloom\CMS\Core\Helpers
 *
 *          --- Model ---
 * @property int                  id
 * @property string               $titre
 * @property string               $code
 * @property string               $requetes
 * @property string               $creation_titre
 * @property string               $route_edit
 * @property string               $on
 * @property string               $s_actions
 * @property string               $s_sub_actions
 * @property string               $s_headers
 * @property int                  $module_id
 * @property int                  $ordre
 *           --- Computed ---
 * @property Actions              $actions
 * @property Actions              $sub_actions
 * @property TableHeader[]        $headers
 * @property Model[]|Collection   $contents
 * @property int                  $number
 *           --- Relation ---
 * @property WorkflowTransition[] $transitions
 * @property Module               $module
 *
 * @method static Builder|static isModule(Module|int $module)
 */
class Workflow extends Model
{
    public function transitions(): HasMany
    {
        return $this->hasMany(WorkflowTransition::class, 'from');
    }

    public function module(): BelongsTo
    {
        return $this->belongsTo(Module::class);
    }

    /**
     * @param Builder $query
     * @param         $module
     * @return Builder
     */
    public function scopeIsModule(Builder $query, $module): Builder
    {
        if ($module instanceof Module) {
            return $query->where('module_id', '=', $module->id);
        }

        return $query->where('module_id', '=', $module);
    }

    /**
     * @param Workflow $workflow
     * @param Model    $model
     *
     * @return bool
     */
    public function legal(Workflow $workflow, Model $model): bool
    {
        if (get_class($model) !== $this->on) {
            return false;
        }

        foreach ($this->transitions as $transition) {
            if ((int)$workflow->id === (int)$transition->to) {
                return true;
            }
        }

        return false;
    }

    public function getActionsAttribute(): Actions
    {
        return (string)$this->s_actions !== '' ? unserialize($this->s_actions) : new Actions();
    }

    public function setActionsAttribute(Actions $value): void
    {
        $this->s_actions = serialize($value);
    }

    public function getSubActionsAttribute(): Actions
    {
        return (string)$this->s_sub_actions !== '' ? unserialize($this->s_sub_actions) : new Actions();
    }

    public function setSubActionsAttribute(Actions $value): void
    {
        $this->s_sub_actions = serialize($value);
    }

    public function getHeadersAttribute(): array
    {
        return (string)$this->s_headers !== '' ? unserialize($this->s_headers) : [];
    }

    public function setHeadersAttribute(array $value): void
    {
        foreach ($value as $item) {
            if (!$item instanceof TableHeader) {
                // On sauvegarde pas les actions si le tableau est valorisé par autre chose que des Actions
                return;
            }
        }
        $this->s_headers = serialize($value);
    }

    /**
     * @return Collection
     */
    public function getContentsAttribute(): Collection
    {
        $class = $this->on;

        /**
         * @var Model $model
         */
        $model = new $class();

        /**
         * @var Model[] $contents
         */
        return $model->newModelQuery()->fromQuery($this->requetes);
    }

    public function getNumberAttribute(): int
    {
        return $this->getConnection()->getPdo()->query($this->requetes)->rowCount();
    }
}
