<?php

/**
 * Created by IntelliJ IDEA.
 * User: loic
 * Date: 01/12/2020
 * Time: 11:04
 */

namespace Bloom\CMS\Core\Http\Events\Handler;

use Bloom\CMS\Core\Contenus\Statut;
use Bloom\CMS\Core\Http\Events\PageArchived;
use Bloom\CMS\Core\Http\Events\PageDeleted;
use Bloom\CMS\Core\Http\Events\PagePublished;
use Bloom\CMS\Core\Http\Events\PageStatutChange;
use Bloom\CMS\Core\Http\Events\PageUnpublished;
use Bloom\CMS\Core\Http\Events\PageUrlChange;
use Bloom\CMS\Core\Http\Page;
use Bloom\CMS\Core\Http\Redirect\Redirection;
use Illuminate\Support\Facades\Event;

class PageHandler
{
    public function statutChanged(PageStatutChange $event)
    {
        $page = $event->page;
        // Si la page est dépublié on crée une redirection vers le dossier parent
        // Si une redirection existe déjà pour le couple ['source','cible'] on publie la redirection
        // Si la page deviens publié on passe la redirection en statut 2
        /**
         * @var Redirection|null $redirection
         */
        $cible = $page->dossier ? $page->dossier->full_pathname : '/';
        if ($page->is_index) {
            $cible = $page->dossier->parent ? $page->dossier->parent->full_pathname : '/';
        }
        $redirection = Redirection::query()
            ->where('source', '=', $page->pathname)
            ->where('cible', '=', $cible)
            ->first();
        if (!$page->is_published) {
            // On crée / publie la redirection
            if ($redirection === null) {
                $redirection = new Redirection();
                $redirection->setStatusChange($page);
                $redirection->save();
                $redirection->bindToSystem();
            }
            $redirection->statut_id = Statut::PUBLIE;
            $redirection->save();
        } else {
            $this->bubbleUp($page);

            if ($redirection !== null) {
                // On dépublie la redirection si elle existe
                $redirection->statut_id = Statut::EN_COURS;
                $redirection->save();
            }
        }
    }

    /**
     * Publie la pages parente si elle existe
     * @param Page $page
     */
    protected function bubbleUp(Page $page): void
    {
        if ($page->dossier) {
            // On republie la page parente si elle existe
            $parent = $page->dossier->index;
            if ($page->is_index) {
                // Si c'est un index on va cherché l'index du dossier parent si il existe
                if ($page->dossier->parent) {
                    $parent = $page->dossier->parent->index;
                } else {
                    $parent = null;
                }
            }
            if ($parent !== null) {
                $parent->statut_id = Statut::PUBLIE;
                $parent->save();
            }
        }
    }

    public function urlChanged(PageUrlChange $event)
    {
        $page = $event->page;
        // Si le pahtname a changé il faut crée des redirects
        // 301 vers la nouvelle url
        $query = Redirection::query()->where('source', '=', $page->getOriginal('pathname'));
        if ($page->is_published || $query->exists()) {
            // Si on a une redirection de publication/dépublication
            // qui existe on la supprime pour faire le ménage
            if ($query->exists()) {
                $query->delete();

                // On crée la redirection vers le dossier parents
                $redirection = new Redirection();
                $redirection->setStatusChange($page);
                if ($page->is_published) {
                    $redirection->statut_id = Statut::EN_COURS;
                }
                $redirection->save();
                $redirection->bindToSystem();
            }

            $redirection = new Redirection();
            $redirection->setPathChange($page);
            $redirection->save();
            // On bind la redirection à une redirection System
            $redirection->bindToSystem();
        }
    }

    public function saving(Page $page)
    {
        //Maj de la métadecription
        $page->changeMetaDescription();

        // Si les champs "slug" et "dossier" son mis à jour on mets à jour le pathname
        $page->updatePathname();

        if (!$page->exists) {
            // Si la page n'existe pas, il s'agit d'une nouvelle page et donc
            // aucune redirection n'est à écrire ni aucun evenement de changement de statut
            return;
        }

        if ($page->isDirty('pathname')) {
            Event::dispatch(new PageUrlChange($page));
        }

        if ($page->isDirty('statut_id')) {
            Event::dispatch(new PageStatutChange($page));
        }

        if ($page->contenu) {
            // On trigger les events de publication / dépublication de page
            switch ((int)$page->statut_id) {
                case Statut::SUPPRIMER:
                    Event::dispatch(new PageDeleted($page));
                    break;
                case Statut::ARCHIVE:
                    Event::dispatch(new PageArchived($page));
                    break;
                case Statut::EN_COURS:
                    if ($page->getOriginal('statut_id') == Statut::PUBLIE) {
                        Event::dispatch(new PageUnpublished($page));
                    }
                    break;
                case Statut::PUBLIE:
                    Event::dispatch(new PagePublished($page));
            }
        }
    }
}
