<?php

namespace Bloom\Model;

use Slim\PDO\Database;
use Slim\PDO\Statement\SelectStatement;

/**
 * Base class to connect and fetch data
 */
abstract class Base implements Model
{
    /**
     * The database driver
     * @var Database
     */
    static protected $db;

    /**
     * The corresponding table in database
     * @var string
     */
    static protected $table;
    /**
     * Association between property and other model
     * @var Relationship[]
     */
    static protected $relationship = [];
    /**
     * List of exclude properties while saving
     * @var array
     */
    static protected $doNotStore = [];
    /**
     * List of exclude properties
     * @var array
     */
    protected static $exclude_all = [];
    /**
     * List of exclude properties
     * @var array
     */
    protected static $exclude_single = [];
    /**
     * @var int
     */
    public $id;

    /**
     * Get items filter by the given where. Will automatically be cast in the current type.
     * So an array of entity will be returns
     * @param array $where Needle form -> [ ['column'=>'foo', 'operator'=>'=', 'value=>'bar'] ]
     * @param string|null $order
     * @param array $select
     * @return static[]
     */
    public static function getItems(array $where = [], $order = null, array $select = [])
    {
        $sth = static::$db
            ->select($select)
            ->from(static::getTable());
        if ($order !== null) {
            $sth->orderBy($order);
        }
        if (count($where) !== 0) {
            foreach ($where as $condition) {
                static::addWhere($sth, $condition);
            }
        }
        /**
         * @var static[] $models
         */
        $models = $sth
            ->execute()
            ->fetchAll(\PDO::FETCH_CLASS, static::class);
        return $models;
    }

    /**
     * @param array $where
     * @param null $order
     * @param array $exclude
     * @return string
     */
    public static function getItemsJson(array $where = [], $order = null, array $exclude = [])
    {
        $models = static::getItems($where, $order);
        $b = [];
        foreach ($models as $model) {
            $model->cleanBuild();
            $b[] = $model->toJSON($exclude, true);
        }
        return '[' . implode(',', $b) . ']';
    }

    /**
     * Get items filter by the given where. Will automatically be cast in the current type.
     * So an array of entity will be returns
     * @param array $where Needle form -> [ ['column'=>'foo', 'operator'=>'=', 'value=>'bar'] ]
     * @param string|null $order
     * @param array $select
     * @return static
     */
    public static function getItem(array $where = [], $order = null, array $select = [])
    {
        $sth = static::$db
            ->select($select)
            ->from(static::getTable());
        if ($order !== null) {
            $sth->orderBy($order);
        }
        if (count($where) !== 0) {
            foreach ($where as $condition) {
                static::addWhere($sth, $condition);
            }
        }
        $model = $sth
            ->execute()
            ->fetchObject(static::class);
        if ($model === false) {
            return false;
        }
        return $model;
    }

    static public function __callStatic($name, $arguments)
    {
        if(preg_match('/^getBy(\w+)/',$name,$matches)) {
            return static::getItem([
                ['column'=>strtolower($matches[1]), 'operator'=>'=', 'value'=>$arguments[0]]
            ], array_key_exists(1,$arguments)?$arguments[1]:null, array_key_exists(2,$arguments)?$arguments[2]:[]);
        }
    }

    /**
     * @param SelectStatement $statement
     * @param array $condition
     */
    protected static function addWhere(SelectStatement $statement, array $condition)
    {
        if (in_array($condition['operator'], ['=', '!=', '<', '>', '<=', '>='], true)) {
            $statement->where($condition['column'], $condition['operator'], $condition['value']);
        } else {
            switch ($condition['operator']) {
                case 'LIKE' :
                    $statement->whereLike($condition['column'], $condition['value']);
                    break;
                case 'NOT LIKE':
                    $statement->whereNotLike($condition['column'], $condition['value']);
                    break;
                case 'IN':
                    $statement->whereIn($condition['column'], $condition['value']);
                    break;
                case 'NOT IN':
                    $statement->whereNotIn($condition['column'], $condition['value']);
                    break;
                default:
                    $statement->whereLike($condition['column'], $condition['value']);
            }
        }
    }

    /**
     * @param null $item
     */
    protected function cleanBuild($item = null)
    {
        foreach (static::$relationship as $relation) {
            if($item === null || $item === $relation->property) {
                $relation->makeItHappen($this, static::$db);
            }
        }
        $this->id = (int)$this->id;
    }

    /**
     * @param $name
     * @return null
     */
    public function __get($name)
    {
        $this->cleanBuild($name);
        if (property_exists($this, $name)) {
            return $this->{$name};
        }
        return null;
    }

    /**
     * @param array $where
     * @param null $order
     * @param array $exclude
     * @return string
     */
    public static function getItemJson(array $where = [], $order = null, array $exclude = [])
    {
        $item = static::getItem($where, $order);
        $item->cleanBuild();
        if ($item) {
            return $item->toJSON($exclude);
        } else {
            return json_encode([
                'error' => [
                    'code' => 404,
                    'message' => 'Item not found'
                ]
            ]);
        }
    }

    /**
     * @param array $exclude
     * @param bool $single
     * @return string
     */
    public function toJSON(array $exclude = [], $single = false)
    {
        $exclude = array_merge($exclude, ($single ? static::$exclude_all : static::$exclude_single));
        foreach ($exclude as $item) {
            $this->{$item} = null;
        }
        return json_encode($this);
    }

    /**
     * @param Database $database
     */
    public static function setDataBase(Database $database)
    {
        static::$db = $database;
    }

    public static function getTable()
    {
        $class = strtolower((new \ReflectionClass(static::class))->getShortName());
        if(!preg_match('/s$/',$class))
        {
            $class.='s';
        }
        return static::$table ?: $class;
    }

    public function save($force = true)
    {
        $reflection = new \ReflectionClass(static::class);
        if($this->id !== 0 && !$force) {

            $stored = static::getItem([['column'=>'id','operator'=>'=','value'=>$this->id]]);
            $toStore = [];
            foreach ($reflection->getProperties(\ReflectionProperty::IS_PUBLIC) as $item) {
                if(
                    $stored->{$item->getName()} !== $this->{$item->getName()} &&
                    (
                        !is_object($this->{$item->getName()}) ||
                        !is_object($stored->{$item->getName()})
                    )
                ) {
                    $toStore[$item->getName()] = $this->{$item->getName()};
                }
            }

            if(count($toStore) > 0) {
                try {
                    static::$db
                        ->update($toStore)
                        ->table(static::getTable())
                        ->where('id','=',$this->id)
                        ->execute()
                    ;
                } catch (\Exception $e) {
                    $_SESSION['error'][] = $e;
                }
            }
        } else {
            $columns = [];
            $values = [];
            foreach ($reflection->getProperties(\ReflectionProperty::IS_PUBLIC) as $property) {
                if(!is_object($this->{$property->getName()})) {
                    foreach(static::$relationship as $relationship) {
                        if($property->getName() === $relationship->property || in_array($property->getName(), static::$doNotStore, true)) {
                            continue 2;
                        }
                    }
                    $columns[] = $property->getName();
                    $values[] = $this->{$property->getName()};
                }
            }
            try {
                $this->id = (int) static::$db->insert($columns)->into(static::getTable())->values($values)->execute();
            } catch(\Exception $e) {
                $_SESSION['error'][] = $e;
            }
        }
    }
}