<?php
/**
 * Created by IntelliJ IDEA.
 * User: bloom
 * Date: 13/01/2016
 * Time: 14:06
 */

namespace Bloom;

/**
 * Class Configuration
 * Parse and read .ini file, if a directory /dev is found next to the ini file and a file with the same name is found
 * it will be parsed and act as an override of the base .ini file
 * @package Bloom
 */
class Configuration
{
    use Singleton;

    /**
     * @var bool
     */
    protected $haveSection = true;

    /**
     * @var array
     */
    protected $conf;
    /**
     * @var array
     */
    protected $override;
    /**
     * @var bool
     */
    protected $haveOverride = false;

    /**
     * @var string
     */
    protected $file;

    /**
     * Configuration constructor.
     * @param null $file
     */
    protected function __construct($file = null)
    {
        $this->file = $file;
        $this->loadConf();
    }

    /**
     * @throws LoadingError when we can not load the configuration file
     */
    protected function loadConf()
    {
        if ($this->file !== null) {
            if (!file_exists($this->file)) {
                throw new LoadingError('File ' . $this->file . ' does not exist');
            }
            $this->conf = parse_ini_file($this->file, $this->haveSection);
            if ($this->conf === false) {
                $error = error_get_last();
                throw new LoadingError($error['message']);
            }
            $dir = dirname($this->file);
            $file = basename($this->file);
            if (file_exists($dir . '/dev/' . $file)) {
                $this->override = parse_ini_file($dir . '/dev/' . $file, $this->haveSection);
                if ($this->override === false) {
                    $error = error_get_last();
                    throw new LoadingError($error['message']);
                }
                $this->haveOverride = true;
            } else {
                $this->haveOverride = false;
            }
        }
    }

    /**
     * @param $name
     * @return mixed|null
     */
    public function __get($name)
    {
        return static::get($name);
    }

    /**
     * @param string $param
     * @param string $section
     * @return mixed|null
     */
    public static function get($param, $section = '')
    {
        $intstance = static::getInstance();
        if ($intstance->haveSection) {
            if (array_key_exists($section, $intstance->conf) && array_key_exists($param, $intstance->conf[$section])) {
                if ($intstance->haveOverride && array_key_exists($section, $intstance->override) && array_key_exists($param, $intstance->override[$section])) {
                    return $intstance->override[$section][$param];
                }
                return $intstance->conf[$section][$param];
            }
            if ($section === '' && array_key_exists($param, $intstance->conf[array_keys($intstance->conf)[0]])) {
                if ($intstance->haveOverride && array_key_exists($param, $intstance->override[array_keys($intstance->override)[0]])) {
                    return $intstance->override[$section][$param];
                }
                return $intstance->conf[array_keys($intstance->conf)[0]][$param];
            }
        } else {
            if (array_key_exists($param, $intstance->conf)) {
                if (array_key_exists($param, $intstance->override)) {
                    return $intstance->override[$param];
                }
                return $intstance->conf[$param];
            }
        }
        return null;
    }

    /**
     * @return $this
     * @throws LoadingError
     */
    public function HaveSection()
    {
        $this->haveSection = true;
        $this->loadConf();
        return $this;
    }

    /**
     * @return $this
     * @throws LoadingError
     */
    public function NoSection()
    {
        $this->haveSection = false;
        $this->loadConf();
        return $this;
    }

    /**
     * @param $file
     * @return $this
     * @throws LoadingError
     */
    public function SetFile($file)
    {
        $this->file = $file . '.ini';
        $this->loadConf();
        return $this;
    }

    /**
     * @throws LoadingError
     */
    public static function Reload()
    {
        static::getInstance()->loadConf();
    }
}